<?php

namespace App\Rules;

use Carbon\Carbon;
use App\Models\Employee;
use App\Models\Vacation;
use Illuminate\Contracts\Validation\Rule;

class VacationRequestRule implements Rule
{
    protected $message = "";
    /**
     * Create a new rule instance.
     *
     * @return void
     */
    public function __construct(
        public $employee_id,
        public $from,
        public $to,
    ) {
        //
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param  string  $attribute
     * @param  mixed  $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        $vacation = Vacation::findOrFail($value);
        $employee = Employee::findOrFail($this->employee_id);

        // vacation request requested before the before_request in vacation
        $diff_in_days_before_request = Carbon::parse($this->from)->diffInDays(Carbon::now());
        if ($diff_in_days_before_request < $vacation->before_request) {
            $this->message = "You should request this vacation '$vacation->name' $vacation->before_request day/s before the vacation request.";
            return false;
        }

        // vacation request maximum number of requested days
        $diff_in_days_maximum_days = Carbon::parse($this->from)->diffInDays($this->to);
        if ($diff_in_days_maximum_days > $vacation->maximum_days_request) {
            $this->message = "The maximum requested days per vacation is $vacation->maximum_days_request.";
            return false;
        }

        // employee vacation requests per year
        $employee_vacations = $employee
            ->vacations()
            ->where('vacation_id', $vacation->id)
            ->where('created_at', '>=', Carbon::now()->subYear())
            ->get();

        $totalVacationDays = 0;

        foreach ($employee_vacations as $employee_vacation) {
            $from = Carbon::parse($employee_vacation->from);
            $to = Carbon::parse($employee_vacation->to);
            $totalVacationDays += $from->diffInDays($to);
        }

        if (($totalVacationDays + $diff_in_days_maximum_days ) > $vacation->maximum_balance) {
            $this->message = "You have " . ($vacation->maximum_balance - $totalVacationDays) . " day/s left in this vacation.";
            return false;
        }

        return true;
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return $this->message;
    }
}
