<?php

namespace App\Trait;

use App\Models\Account;
use App\Models\Company;
use App\Models\Setting;
use App\Http\Helper\CompanyHelper;
use App\Http\Services\Account\AccountService;

trait AccountHelperTrait
{
    /**
     * Copies an account to a new company along with its child accounts.
     *
     * @param Account $account The account to be copied.
     * @param Account|null $parentAccount The parent account in the new company (if any).
     */

    //Takes root account and copy it with its children accounts
    private static function copyAccount(Account $account, ?Account $parentAccount = null): void
    {
        // Copy the account and its children
        self::replicateAccountChildren(
            account: $account,
            newAccount: self::replicateAccount(account: $account, parentAccount: $parentAccount)
        );

        return;
    }

    /**
     * Creates a replica of an account for a new company.
     *
     * @param Account $account The original account to be replicated.
     * @param Account|null $parentAccount The parent account in the new company (if any).
     * @return Account The newly created account in the new company.
     */
    private static function replicateAccount(Account $account, ?Account $parentAccount = null): Account
    {
        // Create a copy of the account
        $newAccount = $account->replicate();

        // Reset specific fields for the new company
        $newAccount->amount = 0; // Reset amount to zero
        $newAccount->company_id = CompanyHelper::getId(); // Assign new company ID
        $newAccount->parent_id = $parentAccount?->id; // Set new parent account ID if available
        $newAccount->code = AccountService::generateCode($parentAccount); // Generate new account code

        // Save the new account to the database
        $newAccount->save();
        //Attach model to new account
        // $models = $account->model; // Assuming 'model' is a relationship and can have multiple models

        // foreach ($models as $model) {
        //     $newAccount->model()->attach($model);
        // }
        return $newAccount;
    }

    /**
     * Recursively replicates child accounts for a given account to a new company.
     *
     * @param Account $account The original parent account.
     * @param Account $newAccount The newly created parent account in the new company.
     */
    private static function replicateAccountChildren(Account $account, Account $newAccount): void
    {
        // Iterate over each child of the original account
        foreach ($account->children as $child) {
            // Recursively copy each child account
            self::copyAccount($child, $newAccount);
        }

        return;
    }

    private static function getMainCompanyId(): int
    {
        //get the main company id from settings table
        return
            Company::where(
                'email',
                Setting::where('key', 'MAIN_COMPANY_EMAIL')->value('value')
            )->value('id')
            ?? Company::first()->id;
    }
}
