<?php

namespace App\Trait;

use App\Models\Tax;
use App\Models\Bill;
use App\Models\Branch;
use App\Models\Client;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\BillProduct;
use App\Models\BillService;
use App\Models\ProductUnit;
use App\Models\ProductWarehouse;
use App\Http\Enums\Bill\BillEnum;
use App\Http\Services\Bill\BillProductService;
use App\Http\Services\Bill\BillServiceService;
use App\Models\PaymentMethod;

trait BillTrait
{
    private function calcDiscount(float $price = 0, string $discount_type, float $discount_amount): float
    {
        if ($discount_type == "percentage") {
            $discount  = $price * ($discount_amount / 100);
            $this->discount += $discount;
            return $discount;
        }

        $this->discount += $discount_amount;
        return $discount_amount;
    }

    private function calcRequestedProductValue(float $quantity, float $price): float
    {
        $total = $quantity * $price;
        $this->total += $total;
        return $total;
    }

    private function tax(int $taxId): Tax
    {
        return Tax::findOrFail($taxId);
    }

    private function calcTaxValue(Tax $tax, float $price, float $discount, ?bool $tax_included = false, ?bool $addToTotalTax = true): float
    {
        $totalAfterDiscount =  ($price - $discount);
        if ($tax_included) {

            $totalWithoutTax = $totalAfterDiscount / (100 + $tax->rate) * 100;

            $tax = $totalAfterDiscount - $totalWithoutTax;
        } else {
            $tax = $totalAfterDiscount * ($tax->rate / 100);
        }

        if ($addToTotalTax) {

            $this->tax += $tax;
        }

        return $tax;
    }

    private function calculateRemainingAmount(float $paid_amount): float
    {
        $remaining_amount = (($this->total - ($this->discount + $this->totalDiscountOnTheBill)) + $this->tax) - $paid_amount;
        
        return (float) number_format($remaining_amount, 2, '.', '');
    }

    private function calcQuantityByMainUnit(ProductUnit $productUnit, float $quantity): float
    {
       
        return $productUnit->conversion_factor * $quantity;
    }

    private function prepareValues(array $data, mixed $key): array
    {
        return array_diff_key($data, [$key => 1, 'id' => 2]);
    }

    private function isSalesBill(): bool
    {
        return $this->bill->type === BillEnum::sales->value;
    }

    public static function getBillableClass(string $route_type)
    {
        return $route_type == BillEnum::sales->value ? Client::class : Supplier::class;
    }

    public static function calcTaxOnBill(float $tax_percentage, float $bill_total): float
    {
        return $bill_total * $tax_percentage;
    }

    public static function calcTaxOnProduct(float $tax_percentage, float $product_price): float
    {
        return $product_price * $tax_percentage;
    }

    public function setBill(Bill $bill): self
    {
        $this->bill = $bill;
        return $this;
    }

    private function calculateUpdatedProductValues(float $currentQuantity, float $newQuantityAmount, float $price, ProductWarehouse $productWarehouse, ?float $otherExpenses = 0): array
    {
        if ($this->isSalesBill()) {
            return $this->calculateValuesForSalesBill($currentQuantity, $newQuantityAmount, $price, $productWarehouse);
        }

        return $this->calculateValuesForNonSalesBill($currentQuantity, $newQuantityAmount, $price, $productWarehouse, $otherExpenses);
    }

    private function calculateValuesForSalesBill(float $currentQuantity, float $newQuantityAmount, ?float $price, ProductWarehouse $productWarehouse): array
    {
        $newQuantity = $currentQuantity - $newQuantityAmount;

        return [
            'value'    => $newQuantity * $productWarehouse->price,
            'quantity' => $newQuantity,
            'price'    => $productWarehouse->price,
        ];
    }

    private function calculateValuesForNonSalesBill(float $currentQuantity, float $newQuantityAmount, float $price, ProductWarehouse $productWarehouse, ?float $otherExpenses = 0): array
    {
        $newValue = $productWarehouse->value + ($newQuantityAmount * $price) + $otherExpenses;
        $newQuantity = $currentQuantity + $newQuantityAmount;

        return [
            'value'    => $newValue,
            'quantity' => $newQuantity,
            'price'    => $newValue / $newQuantity,
        ];
    }

    private function getBillProductsIds(): array
    {
        return BillProduct::where('bill_id', $this->bill->id)->pluck('product_id')->toArray();
    }

    private function getBillServicesIds(): array
    {
        return BillService::where('bill_id', $this->bill->id)->pluck('service_id')->toArray();
    }

    private function deleteUnwantedProducts(array $productIds): void
    {
        BillProduct::where('bill_id', $this->bill->id)->whereIn('product_id', $productIds)->delete();
    }

    private function deleteUnwantedServices(array $serviceIds): void
    {
        BillService::where('bill_id', $this->bill->id)->whereIn('service_id', $serviceIds)->delete();
    }

    private function updateProperties(): void
    {
        $billProductServiceCost = isset($this->billProductService) ? $this->billProductService->cost : 0;
        $billProductServiceDiscount = isset($this->billProductService) ? $this->billProductService->discount + $this->billProductService->totalDiscountOnTheBill : 0;
        // $billProductServiceTax = isset($this->billProductService) ? $this->billProductService->tax : 0;
        $billProductServiceTotal = isset($this->billProductService) ? $this->billProductService->total : 0;

        $billServiceServiceCost = isset($this->billServiceService) ? $this->billServiceService->cost : 0;
        $billServiceServiceDiscount = isset($this->billServiceService) ? $this->billServiceService->discount + $this->billServiceService->totalDiscountOnTheBill : 0;
        // $billServiceServiceTax = isset($this->billServiceService) ? $this->billServiceService->tax : 0;
        $billServiceServiceTotal = isset($this->billServiceService) ? $this->billServiceService->total : 0;

        $this->total = $billProductServiceTotal + $billServiceServiceTotal;
        $this->productsCost = $billProductServiceCost;
        $this->servicesCost = $billServiceServiceCost;
        $this->discount = $billProductServiceDiscount + $billServiceServiceDiscount;
    }

    public static function requestHave(string $value, $request): bool
    {
        return isset($request[$value]) && !empty($request[$value]);
    }

    private function updateProductQuantitiesAndDismissalNotices(): void
    {
        if ($this->billWithProducts) {
            $this->billProductService->updateProductsQuantities();
        }


        if ($this->billWithServices && $this->bill->isSalesBill()) {
            $this->billServiceService->updateDismissalNotices();
        }
    }

    private function initializeBillProductService(array $billProductsData, Warehouse $warehouse): void
    {
        $this->billProductService = new BillProductService(
            requestedProductsData: $billProductsData,
            warehouse: $warehouse,
        );

        $this->billProductService->processRequestedProducts();
    }

    private function initializeBillServiceService(array $billServicesData): void
    {
        $this->billServiceService = new BillServiceService(
            requestedServicesData: $billServicesData,
            warehouse: $this->warehouse
        );

        $this->billServiceService->processRequestedServices();
    }

    private static function getWarehouse(int $warehouseId): Warehouse
    {
        return Warehouse::where('id', $warehouseId)->with(['account'])->firstOrFail();
    }

    private static function getBranch(int $branchId): Branch
    {
        return Branch::where('id', $branchId)->firstOrFail();
    }

    private function calcTotalDiscountOnBill($totalBillDiscount, ?bool $addTaxToTotalTax = true)
    {
        $billTotalBeforeTax = isset($this->billProductService) ? $this->billProductService->total : 0;
        $billTotalBeforeTax += isset($this->billServiceService) ? $this->billServiceService->total : 0;

        // services
        if (isset($this->billServiceService)) {
            foreach ($this->billServiceService->billProcessedItemsData as &$processedService) {
                $productShareAmount = $totalBillDiscount * ($processedService['total'] / $billTotalBeforeTax);
                $processedService['from_total_discount'] = $productShareAmount;
                $taxAmount = isset($processedService['tax_id'])
                    ? $this->calcTaxValue(
                        tax: $processedService['tax_id'],
                        price: $processedService['total'],
                        tax_included: $processedService['tax_included'],
                        discount: $processedService['discount'] + $productShareAmount,
                        addToTotalTax: false
                    ) : null;
                if (isset($processedService['tax_id'])) {
                    $tax = $this->calcTaxValue(
                        tax: $processedService['tax_id'],
                        price: $processedService['total'],
                        tax_included: $processedService['tax_included'],
                        discount: $processedService['discount'] + $productShareAmount,
                        addToTotalTax: $addTaxToTotalTax,
                    );

                    if ($processedService['tax_included']) {
                        $this->billServiceService->total -= $tax;
                    }
                }
                $processedService['tax_amount'] = $taxAmount;
                $this->totalDiscountOnTheBill += $productShareAmount;
            }
        }
        // products
        if (isset($this->billProductService)) {
            foreach ($this->billProductService->billProcessedItemsData as &$processedProduct) {
                $productShareAmount = $totalBillDiscount * ($processedProduct['total'] / $billTotalBeforeTax);
                $processedProduct['from_total_discount'] = $productShareAmount;

                if (isset($processedProduct['tax_id'])) {
                    $tax = $this->calcTaxValue(
                        tax: $processedProduct['tax_id'],
                        price: $processedProduct['total'],
                        tax_included: $processedProduct['tax_included'],
                        discount: $processedProduct['discount'] + $productShareAmount,
                        addToTotalTax: $addTaxToTotalTax,
                    );

                    if ($processedProduct['tax_included']) {
                        $this->billProductService->total -= $tax;
                    }
                }

                $this->totalDiscountOnTheBill += $productShareAmount;
            }
        }
    }

    public function addProcessedItem($processedProduct)
    {
        $product = $processedProduct['item'];
        if (!$this->productExists($product)) {
            $this->billProcessedItemsData[] = $processedProduct;
            return true;
        } else {
            return false;
        }
    }

    private function productExists($product)
    {
        $existingProducts = array_column($this->billProcessedItemsData, 'item');
        return in_array($product, $existingProducts);
    }

    private function getBillOtherExpenses(array $otherExpensesData): self
    {
        $total = 0;
        $otherExpenses = [];

        foreach ($otherExpensesData as $otherExpense) {
            $total += $otherExpense['amount'];

            $otherExpenses[] = [
                'account_id' => $otherExpense['account_id'],
                'amount' => $otherExpense['amount']
            ];
        }
        $this->bill->update(['total_other_expenses_amount' => $total]);

        $this->otherExpensesTotal = $total;

        // $this->otherExpenses = $otherExpenses;
        $this->otherExpenses = $otherExpenses;

        return $this;
    }
    private function getExpensesPaymentMethod($paymentMethodData): self
    {
        $paymentMethod = PaymentMethod::findOrFail($paymentMethodData['payment_method']);
        $this->otherExpensesPaymentMethodAccount = $paymentMethod->model()->find($paymentMethodData['id'])->account;
        return $this;
    }

    private  function calcTax($price, $tax, $tax_included, $quantity)
    {
        if ($tax_included) {
            $this->tax += ($price * $tax->rate) / (100 + $tax->rate) * $quantity;
        } else {
            $this->tax += $price * ($tax->rate / 100) * $quantity;
        }
    }
}
