<?php

namespace App\Trait;

use App\Models\Product;
use Illuminate\Support\Arr;
use App\Trait\UploadFileTrait;
use App\Models\ProductWarehouse;
use App\Http\Helper\CompanyHelper;
use App\Models\ProductBranch;

trait ProductTrait
{
    use UploadFileTrait;

    private function createProductDetails(array $data)
    {
        $image = isset($data['image']) ? $this->uploadFile(Product::UPLOADED_FILES, $data['image']) : null;

        $product = Product::create(
            [
                'image' => $image,
                'code' => $data['code'],
                'company_id' => CompanyHelper::getId(),
            ] + $data
        );

        return $product;
    }

    private function updateProductDetails(Product $product, array $data)
    {
        $image = isset($data['image']) ? $this->uploadFile(Product::UPLOADED_FILES, $data['image']) : $product->image;

        $product->update(
            [
                'image' => $image,
            ] + $data
        );
    }

    private function handleProductUnitsAndWarehouses($request, Product $product): void
    {
        $validatedData = $request->validated();

        if (isset($validatedData['product_unit'])) {
            $this->handleProductUnits($validatedData['product_unit'], $product);
        }

        if (isset($validatedData['warehouse_unit'])) {
            $this->handleProductWarehousesWithUnits($validatedData['warehouse_unit'], $product);
        }
        if (isset($validatedData['branch_unit'])) {
            $this->handleProductBranchsWithUnits($validatedData['branch_unit'], $product);
        }

        return;
    }

    private function handleProductUnits(array $data, Product $product)
    {
        $unitIds = [];

        foreach ($data as $unitData) {

            $product->units()->updateOrCreate(
                [
                    'unit_id' => $unitData['unit_id'],
                ],
                [
                    'main' => $unitData['main'] === 'on',
                    'weight' => 0,
                    'virtual_sale' => $unitData['virtual_sale'] === 'on',
                    'virtual_buying' => $unitData['virtual_buying'] === 'on',
                ] + $unitData
            );

            $unitIds[] = $unitData['unit_id'];
        }

        $product->units()->whereNotIn('unit_id', $unitIds)->delete();
    }

    private function handleProductWarehousesWithUnits(array $data, Product $product)
    {
        $warehouseIds = [];

        foreach ($data as $warehouse) {
            $productWarehouse = $product->warehouses()
                ->updateOrCreate(
                    [
                        'warehouse_id' => $warehouse['warehouse_id'],
                    ],
                    Arr::only($warehouse, [
                        'min_quantity',
                        'max_quantity',
                    ])
                );

            $warehouseIds[] = $warehouse['warehouse_id'];


            $this->createWarehouseUnits($warehouse['units'], $productWarehouse);
        }

        $product->warehouses()->whereNotIn('warehouse_id', $warehouseIds)->delete();
    }

    private function handleProductBranchsWithUnits(array $data, Product $product)
    {
        $branchIds = [];

        foreach ($data as $branch) {
            $productBranch= $product->branchs()
                ->updateOrCreate(
                    [
                        'branche_id' => $branch['branche_id'],
                    ],
                    Arr::only($branch, [
                        'min_quantity',
                        'max_quantity',
                    ])
                );

            $branchIds[] = $branch['branche_id'];


            $this->createBranchUnits($branch['units'], $productBranch);
        }

        $product->branchs()->whereNotIn('branche_id', $branchIds)->delete();
    }

    protected function createWarehouseUnits(array $units, ProductWarehouse $productWarehouse)
    {
        $unitsIds = [];

        foreach ($units as $warehouseUnit) {
            $productWarehouse
                ->warehouseUnits()
                ->updateOrCreate(
                    [
                        'unit_id' => $warehouseUnit['unit_id'],
                    ],
                    $warehouseUnit
                );
            $unitsIds[] = $warehouseUnit['unit_id'];
        }

        $productWarehouse->warehouseUnits()->whereNotIn('unit_id', $unitsIds)->delete();
    }
    protected function createBranchUnits(array $units, ProductBranch $productBranch)
    {
        $unitsIds = [];

        foreach ($units as $branchUnit) {
            $productBranch
                ->branchUnits()
                ->updateOrCreate(
                    [
                        'unit_id' => $branchUnit['unit_id'],
                    ],
                    $branchUnit
                );
            $unitsIds[] = $branchUnit['unit_id'];
        }

        $productBranch->branchUnits()->whereNotIn('unit_id', $unitsIds)->delete();
    }
    private static function generateUniqueCode()
    {
        $code = mt_rand(100000, 999999);

        while (Product::where(
            [
                'code' => $code,
                'company_id' => CompanyHelper::getId(),
            ]
        )->exists()) {
            $code = mt_rand(100000, 999999);
        }

        return $code;
    }

    private static function generateUniqueKey()
    {
        $key = mt_rand(100000, 999999);

        while (Product::where(
            [
                'key' => $key,
                'company_id' => CompanyHelper::getId(),
            ]
        )->exists()) {
            $key = mt_rand(100000, 999999);
        }

        return $key;
    }
}
