<?php

namespace App\Http\Controllers\Api\Bill;

use App\Models\Bill;
use App\Trait\ApiTrait;
use App\Http\Helper\CompanyHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\BillResource;
use App\Http\Services\Bill\BillService;
use Illuminate\Validation\ValidationException;
use App\Http\Requests\Api\Bill\StoreBillRequest;
use App\Http\Requests\Api\Bill\UpdateBillRequest;
use App\Http\Services\Bill\BillJournalEntryService;

class BillController extends Controller
{
    use ApiTrait;

    public function __construct()
    {
        $features = [];

        // if (Route::current()->type == 'sales') {
        //     $features[] = Feature::FEATURE_SALES;
        // } elseif (Route::current()->type == 'purchases') {
        //     $features[] = Feature::FEATURE_PURCHASES;
        // }

        // if (!CompanyFeatureHelper::companyHasFeature($features)) {
        //     // throw ValidationException::withMessages(['package' => 'You do not have this feature, please subscribe to a valid package.']);
        // }
    }

    public function index($type, $billType)
    {

        $status = $billType == 'invoice' ? 'Accredited' : 'Draft';
        if ($status == 'Accredited') {
            $this->authorize('view bills');
        } else {
            $this->authorize('view Sales Orders');
        }

        return $this->apiResponse(
            data: BillResource::collection(
                CompanyHelper::getCompany(request())->bills()->where('type', $type)->where('fast_sales', false)->where('status', $status)->when(auth('worker')->check(), function ($query) {
                    $query->where('worker_id', auth('worker')->user()->id);
                })->with([
                    'journals.entries.account',
                    'journals.entries.type',
                    'branch',
                    'warehouse',
                    'delivery',
                    'billPaymentMethod.payment',
                    'billPaymentMethod.paymentMethod',
                    'sellNature.selling',
                    'billProducts.product',
                    'billServices.service',
                    'billProducts.tax',
                    'billServices.tax',
                    'billProducts.unit',
                    'billable',
                    'billExpenses.expense',
                ])->get()
            )
        );
    }

    public function getFastSalesInvoices()
    {
        $this->authorize('view fast bill sales');
        return $this->apiResponse(
            data: BillResource::collection(
                CompanyHelper::getCompany(request())->bills()->where('fast_sales', true)->when(auth('worker')->check(), function ($query) {
                    $query->where('worker_id', auth('worker')->user()->id);
                })->with([
                    'journals.entries.account',
                    'journals.entries.type',
                    'branch',
                    'warehouse',
                    'delivery',
                    'billPaymentMethod.payment',
                    'billPaymentMethod.paymentMethod',
                    'sellNature.selling',
                    'billProducts.product',
                    'billServices.service',
                    'billProducts.tax',
                    'billServices.tax',
                    'billProducts.unit',
                    'billable',
                    'billExpenses.expense',
                ])->get()
            )
        );
    }

    public function show($type, $billType, Bill $bill)
    {
        if ($bill->fast_sales) {
            $this->authorize('show fast bill sales');
        } else {
            if ($bill->isAccredited()) {
                $this->authorize('show bills');
            } else {
                $this->authorize('show Sales Orders');
            }
        }

        $bill->load([
            'journals.entries.account',
            'journals.entries.type',
            'branch',
            'warehouse',
            'delivery',
            'billPaymentMethod.payment',
            'billPaymentMethod.paymentMethod',
            'sellNature.selling',
            'billProducts.product',
            'billServices.service',
            'billProducts.tax',
            'billServices.tax',
            'billProducts.unit',
            'billable',
            'billExpenses.expense',
        ]);
        return $this->apiResponse(data: new BillResource($bill));
    }

    public function store(StoreBillRequest $request)
    {
        if ($request->fast_sales) {
            $this->authorize('create fast bill sales');
        } else {
            if ($request->status == 'Accredited') {
                $this->authorize('create bills');
            } else {
                $this->authorize('create Sales Orders');
            }
        }
        dd('after validation');
        DB::transaction(function () use ($request) {
            $billService = (new BillService)->billProcess(request: $request, type: 'store');

            if ($billService->bill->isAccredited()) {
                (new BillJournalEntryService($billService))->createJournalEntries($request->bill_details['status']);
            }
        });

        return $this->apiResponse(message: __('messages.bill_created_successfully'));
    }

    public function update(UpdateBillRequest $request, $type, $billType, Bill $bill)
    {
        if ($bill->fast_sales) {
            $this->authorize('edit fast bill sales');
        } else {
            if ($bill->isAccredited()) {
                throw ValidationException::withMessages([
                    'bill' => [__('messages.bill_cannot_update')]
                ]);
            } else {
                $this->authorize('edit Sales Orders');
            }
        }

        DB::transaction(function () use ($request, $bill) {
            $billService = (new BillService)->setBill($bill)->billProcess(request: $request, type: 'update');

            if ($billService->bill->isAccredited()) {
                (new BillJournalEntryService($billService))->createJournalEntries();
            }
        });

        return $this->apiResponse(message: __('messages.bill_updated_successfully'));
    }


    public function destroy($type, $billType, Bill $bill)
    {
        if ($$bill->fast_sales) {
            $this->authorize('delete fast bill sales');
        } else {
            if ($bill->isAccredited()) {
                $this->authorize('delete bills');
            } else {
                $this->authorize('delete Sales Orders');
            }
        }
        $bill->delete();
        return $this->apiResponse(__('messages.bill_deleted_successfully'));
    }
}
