<?php

namespace App\Http\Controllers\Api\Bill;

use App\Models\Bill;
use App\Trait\ApiTrait;
use App\Http\Helper\CompanyHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\BillResource;
use App\Http\Services\Bill\BillService;
use Illuminate\Validation\ValidationException;
use App\Http\Requests\Api\Bill\StoreBillRequest;
use App\Http\Requests\Api\Bill\UpdateBillRequest;
use App\Http\Services\Bill\BillJournalEntryService;

class BillController extends Controller
{
    use ApiTrait;

    public function index($type, $billType)
    {
        return $this->apiResponse(
            data: BillResource::collection(
                CompanyHelper::getCompany(request())->bills()->where('type', $type)->with([
                    'billPaymentMethod.payment',
                    'billPaymentMethod.paymentMethod',
                    'sellNature.selling',
                    'billable'
                ])->get()
            )
        );
    }

    public function show($type, $billType, Bill $bill)
    {
        $bill->load([
            'journals.entries.account',
            'journals.entries.type',
            'branch',
            'warehouse',
            'delivery',
            'billPaymentMethod.payment',
            'billPaymentMethod.paymentMethod',
            'sellNature.selling',
            'billProducts.product',
            'billServices.service',
            'billProducts.tax',
            'billProducts.unit',
            'billable',
        ]);
        return $this->apiResponse(data: new BillResource($bill));
    }

    public function store(StoreBillRequest $request)
    {
        DB::transaction(function () use ($request) {
            $billService = (new BillService)->billProcess(request: $request, type: 'store');
            if ($billService->bill->isAccredited()) {
                (new BillJournalEntryService($billService))->createJournalEntries();
            }
        });

        return $this->apiResponse(message: "Bill Created Successfully.");
    }

    public function update(UpdateBillRequest $request, $type, $billType, Bill $bill)
    {
        if ($bill->isAccredited()) {
            throw ValidationException::withMessages([
                'bil' => ['You cannot modify an accredited bill.']
            ]);
        }

        DB::transaction(function () use ($request, $bill) {
            $billService = (new BillService)->setBill($bill)->billProcess(request: $request, type: 'update');

            if ($billService->bill->isAccredited()) {
                (new BillJournalEntryService($billService))->createJournalEntries();
            }
        });

        return $this->apiResponse(message: "Bill updated Successfully.");
    }

    public function destroy($type, $billType, Bill $bill)
    {
        $bill->delete();
        return $this->apiResponse("Bill deleted Successfully.");
    }
}
