<?php

namespace App\Http\Controllers\Api\Branch\Warehouse;

use App\Models\Journal;
use App\Models\Product;
use App\Trait\ApiTrait;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use App\Http\Services\Account\OpenBalanceService;
use App\Http\Requests\Api\OpenBalance\StoreOpenBalanceRequest;
use App\Models\ProductWarehouse;

class OpenBalanceController extends Controller
{
    use ApiTrait;

    public function store(StoreOpenBalanceRequest $request)
    {
        DB::beginTransaction();
        $total = 0;
        foreach ($request->products as $requested_product) {
            $product = Product::where('id', $requested_product['product_id'])
                ->with(['warehouses' => function ($query) use ($requested_product) {
                    $query->where('warehouse_id', $requested_product['warehouse_id']);
                }, 'units' => function ($query) use ($requested_product) {
                    $query->where('unit_id', $requested_product['unit_id']);
                }])
                ->first();
            $product_unit = $product->units->first();
            $product_warehouse = $product->warehouses->first();

            if (!$product_warehouse) {
                DB::rollBack();
                throw ValidationException::withMessages([
                    'warehouse_id' => ['Product not attached to this warehouse.']
                ]);
            } elseif (!$product_unit) {
                DB::rollBack();
                throw ValidationException::withMessages([
                    'unit_id' => ['Product does not have this unit.']
                ]);
            }

            if ($product->openBalanceJournal($product_warehouse)->exists()
            ) {
                DB::rollBack();
                throw ValidationException::withMessages([
                    'product_id' => ['Product already has open balance.']
                ]);
            }

            (new OpenBalanceService(
                product: $product,
            ))
                ->updateQuantity($requested_product['quantity'], $requested_product['price']);
            $total += $requested_product['quantity'] * $requested_product['price'];
        }
        OpenBalanceService::createJournal(total: $total, product: $product, warehouse: $product_warehouse);
        DB::commit();
        return $this->apiResponse(message: "All done!");
    }
}
