<?php

namespace App\Http\Controllers\Api\Client;

use App\Models\Client;
use App\Models\Address;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Trait\UploadFileTrait;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ClientResource;
use App\Http\Requests\Api\Client\AddImageRequest;
use App\Http\Requests\Api\Client\StoreClientRequest;
use App\Http\Requests\Api\Client\UpdateClientRequest;

class ClientController extends Controller
{
    use UploadFileTrait, ApiTrait;


    public function index(Request $request)
    {
        return $this->apiResponse(data: ClientResource::collection(
            Client::whereBelongsTo(CompanyHelper::getCompany(request()))
                ->doesntHave('blocked')
                ->when($request->has('special'), fn ($q) => $q->where('special', true), fn ($q) => $q->where('special', false))
                ->with(['entries'])
                ->get()
        ));
    }

    public function show(Client $client)
    {
        $client->load([
            'address.area',
            'favoriteEmployee',
            'title',
            'group',
            'address',
            'entries.journal',
        ]);
        return $this->apiResponse(data: new ClientResource($client));
    }

    public function store(StoreClientRequest $request)
    {
        return $this->apiResponse(
            data: new ClientResource(
                Client::create([
                    'address_id' => Address::create($request->only(['area_id', 'post_code', 'address_line']))->id
                ]
                    + $request->validated())
            ),
            message: 'Client created successfully.'
        );
    }

    public function update(UpdateClientRequest $request, Client $client)
    {
        $client->address->update($request->only(['area_id', 'post_code', 'address_line']));
        $client->update($request->validated());
        return $this->apiResponse(data: new ClientResource($client), message: 'Client updated successfully.');
    }

    public function addImage(AddImageRequest $request, Client $client)
    {
        $image = $this->uploadFile(Client::UPLOAD_PATH, $request->image);
        $client = $client->update(['image' => $image]);
        return $this->apiResponse(data: new ClientResource($client), message: 'Client updated successfully.');
    }

    public function changeStatus(Client $client)
    {
        $client->update([
            'status' => !$client->status->value,
        ]);

        return $this->apiResponse(message: 'Client status changed successfully.');
    }

    public function destroy(Client $client)
    {
        Address::findOrFail($client->address->id)->delete();
        $client->forceDelete();
        return $this->apiResponse(message: 'Client deleted successfully.');
    }
}
