<?php

namespace App\Http\Requests\Api\Bill;

use App\Http\Enums\Bill\BillEnum;
use App\Http\Helper\CompanyHelper;
use App\Http\Services\Bill\BillService;
use App\Rules\ValidSellNatureRule;
use Illuminate\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use App\Http\Services\Bill\BillValidationService;

class BillRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return array_merge(
            BillValidationService::rules(request()->route()->type),
        );
    }

    public function prepareForValidation()
    {
        // dd($this->all(), 'asd');
    }

    protected function withValidator(Validator $validator)
    {
        $this->validatePaymentMethod($validator);

        if ($this->isSalesBill()) {
            $this->validateBillSellNature($validator);
        }

        $this->validateProductsQty($validator);

        $validator->after(function ($validator) {
            $this->validateCustomer($validator);
        });
    }

    protected function validateProductsQty(Validator $validator)
    {
        BillValidationService::validateProductsQty(
            requested_products: $this->bill_products,
            warehouse: $this->bill_details['warehouse_id'],
            type: $this->route()->type,
            is_invoice: $this->isInvoiceBill()
        );
    }

    protected function validateCustomer(Validator $validator)
    {
        $billable_type = BillService::getBillableClass($this->route()->type);

        if (!$billable_type::where([
            'id' => $this->bill_details['customer_id'],
            'company_id' => CompanyHelper::getCompany(request())->id
        ])->exists()) {
            throw ValidationException::withMessages([
                'bill_details.customer_id' => ['Invalid customer.']
            ]);
        }

        $this->merge([
            'bill_details' => array_merge(
                $this->input('bill_details', []),
                [
                    'billable_type' => $billable_type,
                    'billable_id' => $this->bill_details['customer_id'],
                ]
            ),
        ]);
    }

    protected function validatePaymentMethod(Validator $validator)
    {
        $paymentMethods = CompanyHelper::getCompany(request())
            ->paymentMethods()
            ->where('payment_methods.id', $this->bill_payment_method['payment_method'] ?? null)
            ->first();

        if (!$paymentMethods) {
            throw ValidationException::withMessages([
                'bill_payment_method.payment_method' => ['Invalid payment method.']
            ]);
        }

        $validator->addRules($paymentMethods->rule);
    }

    protected function validateBillSellNature(Validator $validator)
    {
        $validator->addRules([
            'bill_sell_nature.bill_sell_nature' => [
                'required',
                'exists:main_sell_natures,id',
                new ValidSellNatureRule,
            ],
        ]);

        $sell_nature = CompanyHelper::getCompany(request())
            ->sellNatures()
            ->where('main_sell_natures.id',  $this->bill_sell_nature['bill_sell_nature'] ?? null)
            ->first();

        if (!$sell_nature) {
            throw ValidationException::withMessages([
                'bill_sell_nature.bill_sell_nature' => ['Invalid sell nature.']
            ]);
        }

        $validator->addRules($sell_nature->rule);
    }

    public function isSalesBill()
    {
        return $this->route()->type == BillEnum::sales->value;
    }

    public function isInvoiceBill()
    {
        return $this->route()->billType == 'invoice';
    }
}
