<?php

namespace App\Http\Services\Account;

use App\Models\Level;
use App\Models\Account;
use App\Http\Helper\CompanyHelper;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\ValidationException;


class AccountService
{
    public static Level $nextLevel;

    public static function generateCode($parentAccount = null): string
    {
        $parentCode = $parentAccount ? $parentAccount->code : '';

        static::$nextLevel = static::findAccountNextLevel($parentAccount);

        return self::generateFullCode(self::getAccountOrder($parentAccount), static::$nextLevel->digits) . $parentCode;
    }

    private static function getAccountOrder(?Account $parentAccount = null): string
    {
        if (is_null($parentAccount->id)) {
            return (string)(CompanyHelper::getCompany()->accounts->whereNull('parent_id')->count() + 1);
        }

        return (string) ($parentAccount->children->count() + 1);
    }

    private static function generateFullCode(string $code, int $levelDigits): string
    {
        return str_pad($code, $levelDigits, '0', STR_PAD_RIGHT);
    }

    private static function findAccountNextLevel(?Account $parentAccount = null): Level
    {
        if (is_null($parentAccount->level_id)) {
            return Level::select(['id', 'digits'])->first();
        }

        return Level::where('id', $parentAccount->level_id + 1)->select(['id', 'digits'])->firstOr(
            function () {
                throw ValidationException::withMessages(['parent_id' => 'You can not add deeper level for this account.']);
            }
        );
    }

    public static function deleteAccount($account): void
    {
        if ($account->entries()->first()) {
            throw ValidationException::withMessages(['account' => ['You cant delete this account, It have entries.']]);
        }

        if ($account->status == 'main') {
            $account->forceDelete();
        } else {
            $account->delete();
        }
    }

    public static function getAccountForModel($model, ?int $company_id = null): Account
    {
        $company_id = $company_id ?? CompanyHelper::getCompany(request())->id;

        $account =  Account::where('company_id', $company_id)->whereHas('model', function (Builder $query) use ($model) {
            $query->where('name', $model);
        })->first();

        if (!$account) {
            throw ValidationException::withMessages([
                'account' => ['You must provide account to ' . $model . '.']
            ]);
        }

        return $account;
    }

    public static function updateMultiAccountAmountsWithParentAccounts($accounts_with_amount_types): float
    {
        $total = 0;

        foreach ($accounts_with_amount_types as $accounts_with_amount_type) {
            $operator = self::getOperator(account: $accounts_with_amount_type['account'], entry_type: $accounts_with_amount_type['entry_type']);
            self::updateAccountAmountWithParentAccounts(
                account: $accounts_with_amount_type['account'],
                amount: $accounts_with_amount_type['amount'],
                operator: $operator,
            );
            $total += $accounts_with_amount_type['amount'];
            // $total = self::getTotal($total, $accounts_with_amount_type['amount'], $operator);
        }
        return $total;
    }

    public static function getOperator(Account $account, $entry_type): string
    {
        return $account->type === $entry_type ? '+' : '-';
    }

    private static function updateAccountAmountWithParentAccounts(Account  $account, $amount, $operator = '+'): Account
    {
        $newAmount = self::getNewAmount($account->amount, $amount, $operator);

        $account->update([
            'amount' =>  $newAmount,
        ]);

        self::updateParentAccounts($account, $amount, $operator);

        return $account;
    }

    private static function getNewAmount($currentAmount, $amount, $operator): float
    {
        $new_amount = ($operator === '-') ? ($currentAmount - $amount) : ($currentAmount + $amount);

        // if ($new_amount < 0) { // to allow negative numbers.
        //     throw ValidationException::withMessages(['account' => ['Account amount cannot be negative.']]);
        // }

        return $new_amount;
    }

    private static function updateParentAccounts(Account $account, $newAmount, $operator): void
    {
        $parent = $account->parent;

        while ($parent) {

            $parent->update([
                'amount' => self::getNewAmount($parent->amount, $newAmount, $operator),
            ]);

            $parent = $parent->parent;
        }
    }
}
