<?php

namespace App\Http\Services\Account;

use App\Models\Account;
use App\Models\Journal;
use App\Models\Employee;
use App\Http\Helper\CompanyHelper;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Validation\ValidationException;

class JournalService
{
    public static function createJournal(
        string $date,
        string $type = Journal::AUTO_TYPE,
        ?string $source = null,
        string $description,
        ?string $file = null,
        ?Employee $employee = null,
        bool $status = true,
        array $debit = [],
        array $credit = [],
        ?Model $journalable = null
    ) {
        // Begin a database transaction
        DB::beginTransaction();

        try {
            // Update accounts in the balance sheet
            $debit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($debit);
            $credit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($credit);

            // Check if debit equals credit, otherwise rollback
            if (!self::isBalanced(debitTotal: $debit_total, creditTotal: $credit_total)) {
                throw ValidationException::withMessages(['journal' => [$type . ' ,Debit and credit amounts must be equal.']]);
            }

            // If both sides are zero, commit the transaction and return
            if ($debit_total == 0 || $credit_total == 0) {
                DB::commit();
                return;
            }

            // Create the journal
            $journal_data = [
                'company_id' => CompanyHelper::getCompany(request())->id,
                'date' => $date,
                'type' => $type,
                'source' => $source,
                'description' => $description,
                'file' => $file,
                'employee_id' => optional($employee)->id,
                'status' => $status,
            ];

            // Use the provided relation if available, otherwise create a new journal
            $journal = $journalable ? $journalable->journals()->create($journal_data) : Journal::create($journal_data);

            // Create entries
            self::createEntries(debit: $debit, credit: $credit, journal: $journal);

            // Commit the transaction
            DB::commit();

            return $journal;
        } catch (\Exception $e) {
            // Rollback in case of an exception
            DB::rollBack();
            throw $e;
        }
    }

    protected static function createEntry(Journal $journal, $entry, $type)
    {
        if ($entry['amount'] > 0) {
            $journal->entries()->create([
                'status' => $type,
                'account_id' => $entry['account']->id,
                'type_type' => $entry['type']['type'],
                'type_id' => $entry['type']['id'],
                'amount' => $entry['amount'],
            ]);
        }
    }

    protected static function isBalanced($debitTotal, $creditTotal): bool
    {
        return $debitTotal == $creditTotal || $debitTotal == -$creditTotal;
    }

    protected static function createEntries($debit, $credit, $journal)
    {
        // Create the journal debit entries
        foreach ($debit as $debit_entry) {
            self::createEntry(journal: $journal, entry: $debit_entry, type: 'debit');
        }

        // Create the journal credit entries
        foreach ($credit as $credit_entry) {
            self::createEntry(journal: $journal, entry: $credit_entry, type: 'credit');
        }
    }

    public static function handleSide($data, $side_type)
    {
        $side_data = [];
        $side_amount = 0;
        foreach ($data as $key => $row) {
            $side_data[$key]['account'] =  Account::findOrFail($row['account']);
            $side_data[$key]['type']['type'] = null;
            $side_data[$key]['type']['id'] = null;
            $side_data[$key]['amount'] =  $row['amount'];
            $side_data[$key]['entry_type'] = $side_type;
            $side_amount += $row['amount'];
        }
        return ["side_data" => $side_data, "side_amount" => $side_amount];
    }
}
