<?php

namespace App\Http\Services\Bill;

use App\Models\Bill;
use App\Models\Product;
use App\Trait\BillTrait;
use App\Models\Warehouse;
use App\Models\BillProduct;
use App\Models\ProductUnit;
use App\Models\ProductWarehouse;

class BillProductService
{
    use BillTrait;

    protected array $billProductsData;
    protected Bill $bill;

    public Warehouse $warehouse;
    public $total = 0;
    public $cost = 0;
    public $discount = 0;
    public $tax = 0;

    public function __construct(array $billProductsData, Warehouse $warehouse)
    {
        $this->billProductsData = $billProductsData;
        $this->warehouse = $warehouse;
    }

    public function updateProductsQuantities(): self
    {
        foreach ($this->bill->billProducts as $billProduct) {
            $productWarehouse = ProductWarehouse::where([
                'product_id' => $billProduct->product_id,
                'warehouse_id' => $billProduct->warehouse->id
            ])->first();
            $valueQuantityPrice = $this->calculateUpdatedProductValues(
                price: $billProduct->price,
                currentQuantity: $productWarehouse->quantity,
                productWarehouse: $productWarehouse,
                newQuantityAmount: $this->calcQuantityByMainUnit(
                    productUnit: ProductUnit::where([
                        'unit_id' => $billProduct->unit_id,
                        'product_id' => $billProduct->product_id,
                    ])->first(),
                    quantity: $billProduct->quantity
                )
            );
            $productWarehouse->update([
                'value' => $valueQuantityPrice['value'],
                'quantity' => $valueQuantityPrice['quantity'],
                'price' => $valueQuantityPrice['price'],
            ]);
        }
        return $this;
    }

    public function processRequestedProducts(): self
    {
        $this->cost = 0;
        $this->total = 0;
        $this->discount = 0;
        $this->tax = 0;

        $currentProductIds = isset($this->bill) ? $this->getBillProductsIds() : [];

        foreach ($this->billProductsData as $billProduct) {
            // get product.
            $product = $this->getProductDetails(productId: $billProduct['product_id'], unitId: $billProduct['unit_id']);
            // calc total
            $total = $this->calcRequestedProductValue(
                quantity: $billProduct['quantity'],
                price: $billProduct['price'],
            );
            // calc discount
            $discount = $this->calcDiscount(
                price: $total,
                discount_type: $billProduct['discount_type'],
                discount_amount: $billProduct['discount']
            );
            // calc tax
            if (isset($billProduct['tax_included']) && $billProduct['tax_included'] == '0') {
                $this->calcTaxValue(tax: $this->tax($billProduct['tax_id']), price: $total, discount: $discount);
            }

            if (isset($this->bill)) {
                $this->processBillProduct(product: $product, billProduct: $billProduct);
            }

            unset($currentProductIds[array_search($product->id, $currentProductIds)]);
        }

        isset($this->bill) ? $this->deleteUnwantedProducts($currentProductIds) : null;

        return $this;
    }

    private function processBillProduct(Product $product, array $billProduct)
    {
        $this->createOrUpdateBillProduct([
            'bill_id' => $this->bill->id,
            'product_id' => $product->id,
            'unit_id' => $billProduct['unit_id'],
            'quantity' => $billProduct['quantity'],
            'price' => $billProduct['price'],
            'discount_type' => $billProduct['discount_type'],
            'discount' => $billProduct['discount'],
            'tax_included' => $billProduct['tax_included'],
            'tax_id' => $billProduct['tax_id'] ?? null,
            'warehouse_id' => $this->warehouse->id,
        ]);

        $this->cost += $this->calculateProductCost(product: $product, billProduct: $billProduct);
    }

    private function calculateProductCost(Product $product, array $billProduct): float
    {
        $productUnit = ProductUnit::where([
            'unit_id' => $billProduct['unit_id'],
            'product_id' => $product->id,
        ])->first();

        $quantity = $billProduct['quantity'];

        $warehousePrice = $product->warehouses->first()->price;

        return $this->calcQuantityByMainUnit(productUnit: $productUnit, quantity: $quantity) * $warehousePrice;
    }

    private function getProductDetails(int $productId, int $unitId): Product
    {
        return Product::where('id', $productId)
            ->with([
                'warehouses' => function ($query) {
                    $query->where('warehouse_id', $this->warehouse->id)->select([
                        'id', 'warehouse_id', 'product_id', 'value', 'price', 'quantity'
                    ]);
                },
                'units' => function ($query) use ($unitId) {
                    $query->where('unit_id', $unitId)->select([
                        'id', 'unit_id', 'product_id', 'conversion_factor'
                    ]);
                },
            ])
            ->firstOrFail();
    }

    private function createOrUpdateBillProduct(array $data): BillProduct
    {
        return BillProduct::updateOrCreate([
            'bill_id' => $this->bill->id,
            'product_id' => $data['product_id'],
        ], [
            'unit_id' =>  $data['unit_id'],
            'quantity' =>  $data['quantity'],
            'price' => $data['price'],
            'discount_type' =>  $data['discount_type'],
            'discount' =>  $data['discount'],
            'tax_included' =>  $data['tax_included'],
            'tax_id' =>  $data['tax_id'] ?? null,
            'warehouse_id' => $this->warehouse->id,
        ]);
    }
}
