<?php

namespace App\Http\Services\Bill;

use App\Models\Bill;
use App\Trait\BillTrait;
use App\Models\Warehouse;
use App\Models\PaymentMethod;
use App\Models\BillSellNature;
use App\Models\MainSellNature;
use App\Trait\UploadFileTrait;
use App\Models\BillPaymentMethod;
use App\Http\Helper\CompanyHelper;
use App\Http\Services\Bill\BillServiceService;
use Illuminate\Validation\ValidationException;

class BillService
{
    use UploadFileTrait, BillTrait;

    const REMAIN_VALIDATION_MESSAGE = 'Remain must be greater than or equal to zero.';

    public Bill $bill;
    public Warehouse $warehouse;
    public BillProductService $billProductService;
    public BillServiceService $billServiceService;

    public float $total = 0;
    public float $productsCost = 0;
    public float $servicesCost = 0;
    public float $discount = 0;
    public float $tax = 0;

    public bool $billWithProducts;
    public bool $billWithServices;
    public bool $processTypeUpdate;

    public function billProcess($request, string $type): self
    {
        $this->billWithProducts = self::requestHave(request: $request, value: 'bill_products');
        $this->billWithServices = self::requestHave(request: $request, value: 'services');
        $this->processTypeUpdate = $type === 'update';

        $this->processTypeUpdate ? $this->processProductsServicesOnUpdate() : null;

        $this->warehouse = self::getWarehouse(warehouseId: $request['bill_details']['warehouse_id']);

        if ($this->billWithProducts) {
            $this->initializeBillProductService(billProductsData: $request['bill_products'], warehouse: $this->warehouse);
        }

        if ($this->billWithServices) {
            $this->initializeBillServiceService(billServicesData: $request['services']);
        }

        $this->updateProperties();

        $this->manageBill(billData: $request->bill_details)
            ->manageBillPayment(billPaymentData: $request->validated()['bill_payment_method']);

        if ($this->billWithServices) {
            $this->billServiceService->setBill(bill: $this->bill)->processRequestedServices();
        }

        if ($this->billWithProducts) {
            $this->billProductService->setBill(bill: $this->bill)->processRequestedProducts();
        }

        if ($request->isSalesBill()) {
            $this->manageBillSellNature(sellingNatureData: $request->validated()['bill_sell_nature']);
        }

        if ($this->bill->isAccredited()) {
            $this->updateProductQuantitiesAndDismissalNotices();
        }

        return $this;
    }

    public function manageBillPayment(array $billPaymentData): self
    {
        $payment_method = PaymentMethod::find($billPaymentData['payment_method']);
        $payment_id = $billPaymentData['id'];
        $values =  $this->prepareValues(data: $billPaymentData, key: 'payment_method');

        $paymentData = [
            'bill_id' => $this->bill->id,
            'payment_method_id' => $payment_method->id,
            'payment_type' => $payment_method->model,
            'payment_id' => $payment_id,
            'resource' => $payment_method->resource,
            'value' => $values,
        ];

        if ($this->processTypeUpdate) {
            BillPaymentMethod::updateOrCreate(
                ['bill_id' => $this->bill->id],
                $paymentData
            );
        } else {
            BillPaymentMethod::create($paymentData);
        }

        return $this;
    }

    public function manageBillSellNature(array $sellingNatureData): self
    {
        $sellNature = MainSellNature::find($sellingNatureData['bill_sell_nature']);
        $selling_id = $sellingNatureData['id'] ?? null;
        $values =  $this->prepareValues(data: $sellingNatureData, key: 'bill_sell_nature');

        $sellNatureData = [
            'bill_id' => $this->bill->id,
            'main_sell_nature_id' => $sellNature->id,
            'resource' => $sellNature->resource,
            'model' => $sellNature->model,
            'selling_type' => $sellNature->model,
            'selling_id' => $selling_id,
            'value' => $values,
        ];

        if ($this->processTypeUpdate) {
            BillSellNature::updateOrCreate(
                ['bill_id' => $this->bill->id],
                $sellNatureData
            );
        } else {
            BillSellNature::create($sellNatureData);
        }

        return $this;
    }

    public function manageBill(array $billData): self
    {
        $remain = $this->calculateRemainingAmount(paid_amount: $billData['amount_paid']);

        if ($remain < 0) {
            throw ValidationException::withMessages([
                'bill_details.amount_paid' => [self::REMAIN_VALIDATION_MESSAGE]
            ]);
        }

        $file = isset($billData['attachment_file']) ? $this->uploadFile(Bill::FILE_PATH, $billData['attachment_file'], $this->processTypeUpdate ? $this->bill->attachment_file : null) : null;

        $billAttributes = [
            'remain' => $remain,
            'tax_amount' => $this->tax,
            'before_discount' => $this->total + $this->tax,
            'total_discount' => $this->discount,
            'attachment_file' => $file,
        ] + $billData;

        if ($this->processTypeUpdate) {
            $this->bill->update($billAttributes);
        } else {
            $this->bill = CompanyHelper::getCompany(request())->bills()->create($billAttributes);
        }

        return $this;
    }

    public function processProductsServicesOnUpdate(): self
    {
        $this->bill->billProducts()->delete();
        $this->bill->billServices()->delete();
        return $this;
    }
}
