<?php

namespace App\Http\Services\Bill;

use App\Models\Bill;
use App\Models\Service;
use App\Trait\BillTrait;
use App\Models\Warehouse;
use App\Models\BillService;
use App\Models\ProductUnit;
use App\Models\DismissalNotice;
use App\Models\ProductServiceUnit;
use Illuminate\Validation\ValidationException;

class BillServiceService
{
    use BillTrait;

    private array $billServicesData;
    private array $services;
    private array $dismissalNoticesData = [];
    protected Bill $bill;
    protected Warehouse $warehouse;

    public $total = 0;
    public $cost = 0;
    public $discount = 0;
    public $tax = 0;

    public function __construct(array $billServicesData, Warehouse $warehouse)
    {
        $this->billServicesData = $billServicesData;
        $this->warehouse = $warehouse;
    }

    public function updateDismissalNotices(): self
    {
        foreach ($this->dismissalNoticesData as $dismissalNotice) {
            $total_used_quantities = array_sum($dismissalNotice['used_quantity']);

            $newQuantity = $dismissalNotice['dismissalNotice']->use_quantity - $total_used_quantities;

            if ($newQuantity < 0) {
                throw ValidationException::withMessages([
                    'service_id' => ['This service has products that have no quantity left. Please check service product quantities and create a dismissal notice.']
                ]);
            }

            $dismissalNotice['dismissalNotice']->decrement('use_quantity', $total_used_quantities * 100);
        }

        return $this;
    }

    public function processRequestedServices(): self
    {
        $this->cost = 0;
        $this->total = 0;
        $this->discount = 0;
        $this->tax = 0;

        $currentServiceIds = isset($this->bill) ? $this->getBillServicesIds() : [];

        foreach ($this->billServicesData as $billService) {
            $service = self::getServiceDetails($billService);


            $this->checkProductsHaveAvailableDismissalNotices(
                productIds: $service->productServiceUnits->pluck('product_id')->toArray(),
                branchId: $billService['branch_id'],
            );

            $discount = $this->calcDiscount(
                discount_type: $billService['discount_type'],
                discount_amount: $billService['discount_amount'],
                price: $billService['price']
            );

            $this->calcProductsCost(
                quantity: $billService['quantity'],
                service: $service
            )
                ->calcTotal(
                    price: $billService['price'],
                    quantity: $billService['quantity'],
                    discount: $discount
                );

            if (isset($billService['tax_included']) && $billService['tax_included'] == '0') {
                $this->calcTaxValue(tax: $this->tax($billService['tax_id']), price: $billService['price'], discount: $discount);
            }

            $this->services[] = $service;

            if (isset($this->bill)) {
                $this->processBillService(service: $service, billService: $billService);
            }

            unset($currentServiceIds[array_search($service->id, $currentServiceIds)]);
        }

        isset($this->bill) ? $this->deleteUnwantedServices($currentServiceIds) : null;

        return $this;
    }

    private function processBillService(Service $service, array $billService): BillService
    {
        return BillService::updateOrCreate([
            'bill_id' => $this->bill->id,
            'service_id' => $service->id,
        ], [
            'branch_id' => $billService['branch_id'],
            'discount_type' => $billService['discount_type'],
            'discount_amount' => $billService['discount_amount'],
            'date' => $billService['date'],
            'quantity' => $billService['quantity'],
            'starts_at' => $billService['starts_at'],
            'tax_included' => $billService['tax_included'],
            'tax_id' => isset($billService['tax_id']) ? $billService['tax_id'] : null,
        ]);
    }

    private function checkProductsHaveAvailableDismissalNotices(array $productIds, int $branchId): self
    {
        foreach ($productIds as $productId) {
            $dismissalNotice = DismissalNotice::where([
                'product_id' => $productId,
                'branch_id' => $branchId,

            ])
                ->orderBy('use_quantity', 'desc')
                ->first();

            if (!$dismissalNotice) {
                throw ValidationException::withMessages([
                    'service_id' => ['Please provide a dismissal note for the service products.']
                ]);
            }

            $this->dismissalNoticesData[$dismissalNotice->product_id]['dismissalNotice'] = $dismissalNotice;
        }

        return $this;
    }

    private function calcProductsCost(int $quantity, Service $service): self
    {
        foreach ($service->productServiceUnits as $productServiceUnit) {
            $this->cost += ($this->calcServiceCost($productServiceUnit, $quantity));
        }

        return $this;
    }

    private function calcTotal(float $price, int $quantity, float $discount): self
    {
        $total = $price * $quantity;

        if ($discount > $total) {
            throw ValidationException::withMessages([
                'discount' => ['You cannot set a discount that is greater than the total of the service.']
            ]);
        }

        $this->total += $total;

        return $this;
    }

    private function calcServiceCost(ProductServiceUnit $productServiceUnit, int $quantity): float
    {
        $serviceProductUnit = ProductUnit::where([
            'unit_id' => $productServiceUnit?->company_unit_id,
            'product_id' => $productServiceUnit?->product_id,
        ])->first();

        $conversion_factor = $serviceProductUnit ? $serviceProductUnit->conversion_factor : 1;

        $productCostPrice = $productServiceUnit?->product->warehouses->first()->price;

        $used_quantity = ($productServiceUnit->percentage / 100) * $quantity * $conversion_factor;

        $cost = $used_quantity * $productCostPrice;

        $this->dismissalNoticesData[$productServiceUnit?->product_id]['used_quantity'][$productServiceUnit->service_id] = $used_quantity;

        return $cost;
    }

    private static function getServiceDetails(array $billService): Service
    {
        return Service::where([
            'id' => $billService['service_id'],
        ])
            ->whereHas('branches', fn ($q) => $q->where('branch_id', $billService['branch_id']))
            // ->whereHas('productServiceUnits.product.warehouses', fn ($q) => $q->where('warehouse_id', $billService['warehouse_id']))
            ->with([
                'productServiceUnits:id,product_id,service_id,company_unit_id,percentage',
                'productServiceUnits.product:id',
                'productServiceUnits.product.warehouses'
                => fn ($q) => $q->where('warehouse_id', $billService['warehouse_id'])
                    ->select('id', 'price', 'quantity', 'value', 'product_id')
            ])
            ->firstOrFail(['id', 'price']);
    }
}
