<?php

namespace App\Http\Services;

use App\Http\Helper\CompanyHelper;
use App\Models\Company;
use Illuminate\Support\Facades\DB;

class ProductService
{

    public function create($request)
    {
        return DB::transaction(function () use ($request) {
            // Create the product
            $product = $this->createProductDetails($request->validated()['product_details']);

            // Check if the product is attached to units
            if (isset($request->validated()['product_unit'])) {
                $this->addUnits($request->validated()['product_unit'], $product);
            }

            // Check if the product is attached to a warehouse
            if (isset($request->validated()['product_warehouse'])) {
                $this->addToBranches($request->validated()['product_warehouse'], $product);
            }

            return $product;
        });
    }

    public function update($request, $product)
    {
        return DB::transaction(function () use ($request, $product) {
            // Update the product details
            $this->updateProductDetails($product, $request->validated()['product_details']);

            // Check if the product is attached to units
            if (isset($request->validated()['product_unit'])) {
                $this->updateUnits($request->validated()['product_unit'], $product);
            }

            // Check if the product is attached to a warehouse
            if (isset($request->validated()['product_warehouse'])) {
                $this->updateBranches($request->validated()['product_warehouse'], $product);
            }

            return $product;
        });
    }

    public function createProductDetails($data)
    {
        $product = CompanyHelper::getCompany(request())->products()->create($data);
        return $product;
    }

    public function updateProductDetails($product, $data)
    {
        $product->update($data);
    }

    public function addUnits($data, $product)
    {
        $unitIds = [];

        foreach ($data as $unit) {
            // Check if the unit with the same unit_id already exists
            if (!in_array($unit['unit_id'], $unitIds)) {
                $product->units()->create([
                    'unit_id' => $unit['unit_id'],
                    'main' => $unit['main'] === 'on',
                    'virtual_sale' => $unit['virtual_sale'] === 'on',
                    'virtual_buying' => $unit['virtual_buying'] === 'on',
                ] + $unit);

                $unitIds[] = $unit['unit_id'];
            }
        }
    }

    public function updateUnits($data, $product)
    {
        $unitIdsToUpdate = [];

        foreach ($data as $unitData) {
            $unit = $product->units()->where('unit_id', $unitData['unit_id'])->first();

            if ($unit) {
                // Update the existing unit
                $unit->update([
                    'main' => $unitData['main'] === 'on',
                    'virtual_sale' => $unitData['virtual_sale'] === 'on',
                    'virtual_buying' => $unitData['virtual_buying'] === 'on',
                ] + $unitData);

                $unitIdsToUpdate[] = $unit->id;
            } else {
                // Create a new unit
                $unit = $product->units()->create([
                    'unit_id' => $unitData['unit_id'],
                    'main' => $unitData['main'] === 'on',
                    'virtual_sale' => $unitData['virtual_sale'] === 'on',
                    'virtual_buying' => $unitData['virtual_buying'] === 'on',
                ] + $unitData);

                $unitIdsToUpdate[] = $unit->id;
            }
        }

        // Delete units that were not included in the update
        $product->units()->whereNotIn('id', $unitIdsToUpdate)->delete();
    }

    public function addToBranches($data, $product)
    {
        $warehouseIds = [];

        foreach ($data as $warehouse) {
            // Check if the warehouse with the same warehouse_id already exists
            if (!in_array($warehouse['warehouse_id'], $warehouseIds)) {
                $product->warehouses()->create($warehouse);
                $warehouseIds[] = $warehouse['warehouse_id'];
            }
        }
    }

    public function updateBranches($data, $product)
    {
        $warehouseIdsToUpdate = [];

        foreach ($data as $warehouseData) {
            $warehouse = $product->warehouses()->where('warehouse_id', $warehouseData['warehouse_id'])->first();

            if ($warehouse) {
                // Update the existing warehouse
                $warehouse->update($warehouseData);
                $warehouseIdsToUpdate[] = $warehouse->id;
            } else {
                // Create a new warehouse
                $warehouse = $product->warehouses()->create($warehouseData);
                $warehouseIdsToUpdate[] = $warehouse->id;
            }
        }

        // Delete warehouses that were not included in the update
        $product->warehouses()->whereNotIn('id', $warehouseIdsToUpdate)->delete();
    }
}
