<?php

namespace App\Http\Services\StockTransfer;

use App\Models\Journal;
use App\Models\Product;
use App\Models\StockTransfer;
use App\Models\ProductWarehouse;
use App\Http\Helper\AccountHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use App\Http\Requests\Api\StockTransfer\StoreStockTransferRequest;
use Illuminate\Validation\ValidationException;

class StockTransferService
{

    const DESCRIPTION = "Stock Transfer journal";

    private ProductWarehouse $fromProductWarehouse;
    private ProductWarehouse $toProductWarehouse;
    private StockTransfer $stockTransfer;
    private StoreStockTransferRequest $request;

    private int $productId;
    private int $quantity;

    public function processTransfer(StoreStockTransferRequest $request): self
    {
        return
            DB::transaction(function () use ($request) {
                return $this->setRequest(request: $request)
                    ->updateWarehousesQuantity()
                    ->createStockTransfer()
                    ->createStockTransferJournals();
            });
    }

    private function updateWarehousesQuantity(): self
    {
        $this->updateFromWarehouse()->updateToWarehouse();

        return $this;
    }

    private function updateFromWarehouse(): self
    {
        $fromProductWarehouse = ProductWarehouse::where([
            'warehouse_id' => $this->request->from_warehouse,
            'product_id' => $this->productId,
        ])->firstOrFail();

        if ($fromProductWarehouse->quantity < $this->quantity) {
            throw ValidationException::withMessages([
                'from_warehouse' => ['Quantity not enough.']
            ]);
        }

        $fromNewQuantity = $fromProductWarehouse->quantity - $this->quantity;

        $fromProductWarehouse->update([
            'quantity' =>  $fromNewQuantity,
            'value' =>  $fromNewQuantity / ($fromProductWarehouse->price ? $fromProductWarehouse->price : 1),
        ]);

        $this->fromProductWarehouse = $fromProductWarehouse;

        return $this;
    }

    private function updateToWarehouse(): self
    {
        $toProductWarehouse = ProductWarehouse::where([
            'warehouse_id' => $this->request->to_warehouse,
            'product_id' => $this->productId,
        ])->firstOrFail();

        $toValue = $toProductWarehouse->value + ($this->quantity * ($this->fromProductWarehouse->price ? $this->fromProductWarehouse->price : 1));
        $toQuantity = $toProductWarehouse->quantity + $this->quantity;

        $toProductWarehouse->update([
            'quantity' =>  $toQuantity,
            'value' =>  $toValue,
            'price' =>  $toValue / $toQuantity,
        ]);

        $this->toProductWarehouse = $toProductWarehouse;

        return $this;
    }

    private function createStockTransferJournals(): self
    {
        $total = $this->fromProductWarehouse->price * $this->quantity;

        JournalService::createJournal(
            date: now(),
            type: Journal::AUTO_TYPE,
            source: "Stock Transfer",
            description: self::DESCRIPTION,
            file: null,
            employee: null,
            status: true,
            debit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                    'type' => [
                        'type' => Product::class,
                        'id' => $this->productId,
                    ],
                    'amount' => $total,
                    'entry_type' => "debit",
                ],
            ],
            credit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                    'type' => [
                        'type' => Product::class,
                        'id' => $this->productId,
                    ],
                    'amount' => $total,
                    'entry_type' => "credit",
                ],
            ],
            journalable: $this->stockTransfer
        );

        return $this;
    }

    private function setData(): self
    {
        $this->productId = $this->request->product_id;
        $this->quantity = $this->request->quantity;
        return $this;
    }

    private function createStockTransfer(): self
    {
        $this->stockTransfer = StockTransfer::create($this->request->validated());
        return $this;
    }

    private function setRequest(StoreStockTransferRequest $request): self
    {
        $this->request = $request;
        $this->setData();
        return $this;
    }
}
