<?php

namespace App\Models;

use App\Models\User;
use App\Models\Address;
use App\Http\Helper\AccountHelper;
use App\Http\Resources\ClientResource;
use App\Http\Enums\User\UserStatusEnum;
use Illuminate\Database\Eloquent\Model;
use App\Http\Services\Account\AccountService;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Client extends Model
{
    use HasFactory, SoftDeletes;

    const UPLOAD_PATH = 'uploads/images/clients/';

    const RULES = [
        'company_id' => 'required|exists:companies,id',
        'code' => 'nullable',
        'membership_number' => 'nullable|string',
        'title_id' => 'nullable|exists:attributes,id,model,clients',
        'group_id' => 'nullable|exists:attributes,id,model,clients',
        'name' => 'required|string',
        'id_number' => 'nullable|string',
        'address_id' => 'nullable|integer',
        'birth_date' => 'nullable|date',
        'work_phone' => 'nullable|string',
        'gender' => 'nullable|in:male,female',
        'job_name' => 'nullable|string',
        'favorite_employee_id' => 'nullable|exists:users,id',
        'crn' => 'nullable|string',
        'trn' => 'nullable|string',
        'note' => 'nullable|string',
        // 'status' => 'nullable|boolean',
        'is_client' => 'nullable|boolean',
        'cache' => 'nullable|boolean',
        'internet' => 'nullable|boolean',
        'fingerprint' => 'nullable|boolean',
        'medical' => 'nullable',
        'medical_history' => 'nullable',
        'contact_info' => 'nullable',
        'extra_data' => 'nullable',
    ];

    protected $fillable = [
        'company_id',
        'code',
        'membership_number',
        'title_id',
        'group_id',
        'name',
        'image',
        'phone',
        'email',
        'id_number',
        'address_id',
        'birth_date',
        'work_phone',
        'gender',
        'job_name',
        'favorite_employee_id',
        'crn', // Commercial Registration
        'trn', // Tax registration number
        'note',
        'status',
        'special',
        'is_client',
        'cache',
        'internet',
        'fingerprint',
        'medical',
        'medical_history',
        'contact_info',
        'extra_data',
    ];

    protected $casts = [
        'medical' => 'array',
        'medical_history' => 'array',
        'contact_info' => 'array',
        'extra_data' => 'array',
        'status' => UserStatusEnum::class,
    ];

    public function resolveResource()
    {
        return ClientResource::class;
    }

    public function entries()
    {
        return $this->morphMany(JournalEntry::class, 'type');
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function group()
    {
        return $this->belongsTo(Attribute::class);
    }

    public function title()
    {
        return $this->belongsTo(Attribute::class);
    }

    public function blocked()
    {
        return $this->hasOne(BlockClient::class);
    }

    public function bills()
    {
        return $this->morphMany(Bill::class, 'billable');
    }


    // public function getGenderAttribute()
    // {
    //     return $this->attributes['gender'] === 'male' ? 1 : 2; // 1 for male, 2 for female
    // }


    public function address()
    {
        return $this->belongsTo(Address::class, 'address_id');
    }

    public function favoriteEmployee()
    {
        return $this->belongsTo(User::class, 'favorite_employee_id');
    }

    public function account()
    {
        return AccountService::getAccountForModel(AccountHelper::ACCOUNT_CLIENTS);
    }
}
