<?php

namespace App\Trait;

use App\Http\Enums\Bill\BillEnum;
use App\Http\Helper\AccountHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use App\Http\Services\Bill\BillJournalEntryService;

trait BillJournalEntryTrait
{
    private function getType(): string
    {
        return ($this->billService->bill->type === BillEnum::sales->value) ? BillJournalEntryService::SALES_NAME : BillJournalEntryService::PURCHASE_NAME;
    }

    private function getDescription(): string
    {
        return ($this->billService->bill->type === BillEnum::sales->value) ? BillJournalEntryService::SALES_DESCRIPTION : BillJournalEntryService::PURCHASE_DESCRIPTION;
    }

    private function createSalesJournalEntries(): void
    {
        DB::transaction(function () {
            // An accounting entry proving the cost of the products
            $costEntries = $this->salesCostEntries();
            JournalService::createJournal(
                date: now(),
                type: $this->getType(),
                source: 'Sales bill',
                description: 'proving the cost of the products',
                file: null,
                employee: null,
                status: true,
                debit: $costEntries['debit'],
                credit: $costEntries['credit'],
                journalable: $this->billService->bill
            );

            // An accounting record proving the sale process
            $proofSaleEntires = $this->salesSalesProofEntries();
            JournalService::createJournal(
                date: now(),
                type: $this->getType(),
                source: 'Sales bill',
                description: 'proving the sale process',
                file: null,
                employee: null,
                status: true,
                debit: $proofSaleEntires['debit'],
                credit: $proofSaleEntires['credit'],
                journalable: $this->billService->bill
            );

            // An accounting record proving the payment process
            $proofPaymentEntries = $this->salesProofPayment();
            JournalService::createJournal(
                date: now(),
                type: $this->getType(),
                source: 'Sales bill',
                description: 'proving the payment process',
                file: null,
                employee: null,
                status: true,
                debit: $proofPaymentEntries['debit'],
                credit: $proofPaymentEntries['credit'],
                journalable: $this->billService->bill
            );
        });
    }

    private function createPurchaseJournalEntries(): void
    {
        DB::transaction(function () {
            // An accounting record proving the purchase process
            $purchasesProofEntries = $this->purchaseSalesProofEntries();
            JournalService::createJournal(
                date: now(),
                type: $this->getType(),
                source: 'Purchase bill',
                description: 'proving the purchase process',
                file: null,
                employee: null,
                status: true,
                debit: $purchasesProofEntries['debit'],
                credit: $purchasesProofEntries['credit'],
                journalable: $this->billService->bill
            );
            // An accounting record proving the payment process
            $purchaseProofPayment = $this->purchaseProofPayment();
            JournalService::createJournal(
                date: now(),
                type: $this->getType(),
                source: 'Purchase bill',
                description: 'proving the payment process',
                file: null,
                employee: null,
                status: true,
                debit: $purchaseProofPayment['debit'],
                credit: $purchaseProofPayment['credit'],
                journalable: $this->billService->bill
            );
        });
    }

    private function salesCostEntries(): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        if ($this->billService->billWithProducts) {
            $entries['debit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS_COSTS),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->billService->billProductService->cost,
                    'entry_type' => "debit",
                ];

            $entries['credit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->billService->billProductService->cost,
                    'entry_type' => "credit",
                ];
        }

        if ($this->billService->billWithServices) {
            $entries['debit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SERVICE_PRODUCTS_COSTS),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->billService->billServiceService->cost,
                    'entry_type' => "debit",
                ];

            $entries['credit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIES_WAREHOUSE),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->billService->billServiceService->cost,
                    'entry_type' => "credit",
                ];
        }

        return $entries;
    }

    private function salesSalesProofEntries(): array
    {
        $tax = $this->getTotalTax();
        $discount = $this->getTotalDiscount();

        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_CLIENTS),
                'type' => [
                    'type' => $this->billService->bill->billable_type,
                    'id' => $this->billService->bill->billable_id,
                ],
                'amount' => ($this->getProductsServicesTotal() + $tax) - $discount,
                'entry_type' => "debit",
            ];

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_ALLOWED_DISCOUNT),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $discount,
                'entry_type' => "debit",
            ];
        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SALES_VALUE_TAX),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $tax,
                'entry_type' => "credit",
            ];
        if ($this->billService->billWithProducts) {
            $entries['credit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SALES),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->getProductsTotal(),
                    'entry_type' => "credit",
                ];
        }

        if ($this->billService->billWithServices) {
            $entries['credit'][] =
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SERVICE_SALES),
                    'type' => [
                        'type' => null,
                        'id' => null,
                    ],
                    'amount' => $this->getServicesTotal(),
                    'entry_type' => "credit",
                ];
        }

        return $entries;
    }

    private function salesProofPayment(): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel($this->billService->bill->billPaymentMethod->paymentMethod->account_name),
                'type' => [
                    'type' => $this->billService->bill->billPaymentMethod->payment_type,
                    'id' => $this->billService->bill->billPaymentMethod->payment_id,
                ],
                'amount' => $this->billService->bill->amount_paid,
                'entry_type' => "debit",
            ];

        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_CLIENTS),
                'type' => [
                    'type' => $this->billService->bill->billable_type,
                    'id' => $this->billService->bill->billable_id,
                ],
                'amount' => $this->billService->bill->amount_paid,
                'entry_type' => "credit",
            ];

        return $entries;
    }

    private function purchaseProofPayment(): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIERS),
                'type' => [
                    'type' => $this->billService->bill->billable_type,
                    'id' => $this->billService->bill->billable_id,
                ],
                'amount' => $this->billService->bill->amount_paid,
                'entry_type' => "debit",
            ];

        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel($this->billService->bill->billPaymentMethod->paymentMethod->account_name),
                'type' => [
                    'type' => $this->billService->bill->billPaymentMethod->payment_type,
                    'id' => $this->billService->bill->billPaymentMethod->payment_id,
                ],
                'amount' => $this->billService->bill->amount_paid,
                'entry_type' => "credit",
            ];

        return $entries;
    }

    private function purchaseSalesProofEntries(): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];
        $tax_total = $this->getTotalTax();

        if ($this->billService->billWithServices){
            $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SERVICE_EXPENSES),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $this->getServicesTotal(),
                'entry_type' => "debit",
            ];
        }

        if ($this->billService->billWithProducts){
            $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => ($this->getProductsTotal()),
                'entry_type' => "debit",
            ];
        }

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PURCHASE_VALUE_TAX),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $tax_total,
                'entry_type' => "debit",
            ];

        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIERS),
                'type' => [
                    'type' => $this->billService->bill->billable_type,
                    'id' => $this->billService->bill->billable_id,
                ],
                'amount' => ($this->getProductsServicesTotal() + $tax_total) - $this->getTotalDiscount(),
                'entry_type' => "credit",
            ];

        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_EARNED_DISCOUNT),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $this->getTotalDiscount(),
                'entry_type' => "credit",
            ];

        return $entries;
    }

    private function getTotalTax(): float
    {
        $tax = 0;

        $tax += $this->getServicesTax();

        $tax += $this->getProductsTax();

        return $tax;
    }

    private function getServicesTax(): float
    {
        $tax = 0;

        if ($this->billService->billWithServices) {
            $tax += $this->billService->billServiceService->tax;
        }

        return $tax;
    }

    private function getProductsTax(): float
    {
        $tax = 0;

        if ($this->billService->billWithProducts) {
            $tax += $this->billService->billProductService->tax;
        }

        return $tax;
    }

    private function getTotalDiscount(): float
    {
        $discount = 0;

        $discount += $this->getServicesDiscount();

        $discount += $this->getProductsDiscount();

        return $discount;
    }

    private function getServicesDiscount(): float
    {
        $discount = 0;

        if ($this->billService->billWithServices) {
            $discount += $this->billService->billServiceService->discount;
        }

        return $discount;
    }

    private function getProductsDiscount(): float
    {
        $discount = 0;

        if ($this->billService->billWithProducts) {
            $discount += $this->billService->billProductService->discount;
        }

        return $discount;
    }

    private function getProductsServicesTotal(): float
    {
        $total = 0;

        $total += $this->getServicesTotal();

        $total += $this->getProductsTotal();

        return $total;
    }

    private function getServicesTotal(): float
    {
        $total = 0;

        if ($this->billService->billWithServices) {
            $total += $this->billService->billServiceService->total;
        }

        return $total;
    }

    private function getProductsTotal(): float
    {
        $total = 0;

        if ($this->billService->billWithProducts) {
            $total += $this->billService->billProductService->total;
        }

        return $total;
    }
}
